const fs = require('fs');
const path = require('path');
const matter = require('gray-matter');

/** @type {import('next-sitemap').IConfig} */
module.exports = {
  siteUrl: 'https://elencantodebenasque.com',
  generateRobotsTxt: true,
  generateIndexSitemap: false, // Un único sitemap.xml, sin índice
  changefreq: 'daily',
  priority: 0.7,
  sitemapSize: 50000, // Límite alto para evitar split
  exclude: ['/legal', '/normas', '/privacidad'],
  robotsTxtOptions: {
    policies: [
      { userAgent: '*', allow: '/' },
    ],
    additionalSitemaps: [
      'https://elencantodebenasque.com/sitemap.xml',
    ],
  },
  // Incluye todas las rutas dinámicas del blog desde archivos Markdown
  additionalPaths: async (config) => {
    const postsDirectory = path.join(process.cwd(), 'content/blog');

    if (!fs.existsSync(postsDirectory)) {
      return [];
    }

    const fileNames = fs.readdirSync(postsDirectory);

    const blogPosts = fileNames
      .filter(fileName => fileName.endsWith('.md'))
      .map(fileName => {
        const fullPath = path.join(postsDirectory, fileName);
        const fileContents = fs.readFileSync(fullPath, 'utf8');
        const { data } = matter(fileContents);

        return {
          loc: `/blog/${data.slug}`,
          changefreq: 'weekly',
          priority: 0.7,
          lastmod: data.date,
        };
      });

    // Rutas con prioridades personalizadas
    const customPaths = [
      {
        loc: '/',
        changefreq: 'daily',
        priority: 1.0,
        lastmod: new Date().toISOString(),
      },
      {
        loc: '/blog',
        changefreq: 'daily',
        priority: 0.8,
        lastmod: new Date().toISOString(),
      },
      {
        loc: '/actividades',
        changefreq: 'weekly',
        priority: 0.9,
      },
      {
        loc: '/alrededores',
        changefreq: 'weekly',
        priority: 0.9,
      },
      {
        loc: '/gastronomia',
        changefreq: 'weekly',
        priority: 0.8,
      },
      {
        loc: '/rutas',
        changefreq: 'weekly',
        priority: 0.8,
      },
      {
        loc: '/descuentos',
        changefreq: 'weekly',
        priority: 0.8,
      },
      {
        loc: '/faqs',
        changefreq: 'monthly',
        priority: 0.6,
      },
      {
        loc: '/legal',
        changefreq: 'yearly',
        priority: 0.3,
      },
      {
        loc: '/normas',
        changefreq: 'yearly',
        priority: 0.3,
      },
      {
        loc: '/privacidad',
        changefreq: 'yearly',
        priority: 0.3,
      },
    ];

    return [...customPaths, ...blogPosts];
  },
};
